/******************************************************************************
*                                                                             *
* Freescale Semiconductor Inc.                                                *
* (c) Copyright 2004-2008 Freescale Semiconductor, Inc.                       *
* ALL RIGHTS RESERVED.                                                        *
*                                                                             *
*******************************************************************************
*******************************************************************************
* Example for RS08KA2 - I2C Slave + IR decoder to GPIO port                   *
*                       I2C max speed 100kHz;                                 *
*                       IR - RC-5 frame;                                      *
******************************************************************************/

#include <hidef.h>                      // for EnableInterrupts macro;
#include "derivative.h"                 // include peripheral declarations;
#include "application.h"                // interface to the assembly module;

void _Startup(void);

//*****************************************************************************
void main(void) 
{
    MCU_init();                         // call Device Initialization;

    Init_Variables();
                                        // Enable KBI inputs STBY,IR,SDA,SCL;
    KBIPE = (BUTTON_MASK|RC5_PIN_MASK|IIC_MASK);       

    for(;;) 
    {
        __RESET_WATCHDOG();
        
        if(mode == SLEEP)
        {
            Go_sleep();
            asm stop;                   //  put MCU to STOP mode;
        }

        temp = (~PTAD) & (BUTTON_MASK|RC5_PIN_MASK|IIC_MASK);

        switch (temp)
        {
            case RC5_PIN_MASK:
            {
                IR_receive();           // Receive IR RC-5 code;
                Compare_IR();           // Decode RC-5 message;
                                        // Enable KBI inputs STBY,IR,SDA,SCL;
                KBIPE = (BUTTON_MASK|RC5_PIN_MASK|IIC_MASK);       
            } break;
            
            case BUTTON_MASK:
            {
                STBY_Button();          // Serve the pressed button;
                                        // Enable KBI inputs STBY,IR,SDA,SCL;
                KBIPE = (BUTTON_MASK|RC5_PIN_MASK|IIC_MASK);       
            } break;
            
            case IIC_MASK:
            {
                KBIPE = 0;              // Disable KBI inputs;
                I2C_Read();             // communicate as I2C Slave device;
                Compare_I2C();          // check command for STB PWR OFF;
                                        // Enable KBI inputs STBY,IR;
                KBIPE = (RC5_PIN_MASK|BUTTON_MASK); 
            } break;
            
            default:
            break;
        }
        
        KBISC_KBACK = 1;                // Clear the KBI flag;
        KBISC_KBIE = 1;                 // Enable KBI interrupt;
        
        if(address_valid && command_ON && (PWR_PIN == OFF))
        {
            PWR_PIN = ON;               // switch ON the main power supply;
            mode = RUN;                 // set mode of the SPMU;
        }
        if(address_valid && command_OFF && (PWR_PIN == ON)) 
        {
            PWR_PIN = OFF;              // switch OFF the main power supply;
            mode = SLEEP;               // set mode of the SPMU;
        }

        address_valid = FALSE;
        command_ON = FALSE;
        command_OFF = FALSE;
    }
}
/********************************************************************************
*   Description: Compares the "Address" and "Command" bytes recsived by I2C;    *
*                                                                               *
*   Input:  - "Address" and "Command" bytes;                                    *
*                                                                               *
*   Output: - STB power supply switch OFF or other function;                    *
*                                                                               *
********************************************************************************/
void Compare_I2C(void)
{
    address_valid = FALSE;
    switch (Address)
    {
        case MY_I2C_ADDR:
        {
            address_valid = TRUE;
        } break;
    }
    
    command_OFF = FALSE;
    switch (Command)
    {
        case CMD_OFF1:
        {
            command_OFF = TRUE;
        } break;
    }
}
/********************************************************************************
*   Description: Receives I2C message;                                          *
*                                                                               *
*   Input:  - I2C bus;                                                          *
*                                                                               *
*   Output: - "Address" and "Command" bytes;                                    *
*           - I2C bus - slave sends the PTA contents to master;                 *
*                                                                               *
********************************************************************************/
void I2C_Read(void)
{
asm
    {
    i2c_slave_func:
            lda PTAD;
            and #I2C_MASK;
            cmp #0;                     // right START condition occured;
            bne i2c_slave_func;         // not valid I2C Start condition;
            //--------- start receive first byte --------------------
            clra;
            //--------- receive bit7 ----------------------
    bit17:
            brclr SCLPIN,PTAD,bit17;    // wait for SCL=1, then read SDA;
            brclr SDAPIN,PTAD,bit17a;
            add #$01;                   // read SDA = 1
    bit17a:     
            lsla;                       // read SDA = 0
    w17:
            brset SCLPIN,PTAD,w17;      // wait for SCL = 0;
            //--------- receive bit6 ----------------------
    bit16:
            brclr SCLPIN,PTAD,bit16;
            brclr SDAPIN,PTAD,bit16a;
            add #$01;                   // read SDA = 1
    bit16a:
            lsla;                       // read SDA = 0
    w16:
            brset SCLPIN,PTAD,w16;
            //--------- receive bit5 ----------------------
    bit15:
            brclr SCLPIN,PTAD,bit15;
            brclr SDAPIN,PTAD,bit15a;
            add #$01;                   // read SDA = 1
    bit15a:
            lsla;                       // read SDA = 0
    w15:
            brset SCLPIN,PTAD,w15;
            //--------- receive bit4 ----------------------
    bit14:
            brclr SCLPIN,PTAD,bit14;
            brclr SDAPIN,PTAD,bit14a;
            add #$01;                   // read SDA = 1
    bit14a:
            lsla;                       // read SDA = 0
    w14:
            brset SCLPIN,PTAD,w14;
            //--------- receive bit3 ----------------------
    bit13:
            brclr SCLPIN,PTAD,bit13;
            brclr SDAPIN,PTAD,bit13a;
            add #$01;                   // read SDA = 1
    bit13a:
            lsla;                       // read SDA = 0
    w13:
            brset SCLPIN,PTAD,w13;
            //--------- receive bit2 ----------------------
    bit12:
            brclr SCLPIN,PTAD,bit12;
            brclr SDAPIN,PTAD,bit12a;
            add #$01;                   // read SDA = 1
    bit12a:
            lsla;                       // read SDA = 0
    w12:
            brset SCLPIN,PTAD,w12;
            //--------- receive bit1 ----------------------
    bit11:
            brclr SCLPIN,PTAD,bit11;
            brclr SDAPIN,PTAD,bit11a;
            add #$01;                   // read SDA = 1
    bit11a:
            lsla;                       // read SDA = 0
            sta Address;                // save received 
                                        // Address byte excluding RW bit;
    w11:
            brset SCLPIN,PTAD,w11;
            //--------- receive bit0 ----------------------
    bit10:
            brclr SCLPIN,PTAD,bit10;    // wait for SCL = 1;
            brclr SDAPIN,PTAD,w10;      // read SDA pin;
            inc r_w;                    // read SDA = 1 -> bit RW = 1;
                                        // read SDA = 0 -> bir RW = 0;
    w10:
            brset SCLPIN,PTAD,w10;
            //--------- send SL_ACK -----------------------            
            bclr 4,PTAD;           // SDAPIN;
            bset 4,PTADD;
    sl_ack_1a:
            brclr SCLPIN,PTAD,sl_ack_1a;
            cmp #MY_I2C_ADDR;           // compare valid own IIC address;
            bne go_back;                // go to start if not match;
    sl_ack_1b:
            brset SCLPIN,PTAD,sl_ack_1b;
            bclr SDAPIN,PTADD;          // SDA = PTA1 as input;
            //--------- read the PTA port -----------------
            lda PTAD;                   // read portA
            sta TX_data;                // save data;
            //--------- end of read the PTA port ----------
            brclr 0,r_w,rec_i2c_data;   // go to receive next byte (RX_data);
            bra go_send_data;           // go to send (TX_data=PTA) to Master;
            
            //--------- end of receive first byte ---------
    go_back:    
            rts;                    // not valid device address -> go to back;
    //------------------ first byte received ------------------------            
//-----------------------------------------------------------------------------            
    go_send_data:
            jmp send_data;          // send data from USER_REGISTER to master;
//-----------------------------------------------------------------------------
    //------------------ start receive second byte ------------------
    rec_i2c_data:
            clra;
            //--------- receive bit7 ----------------------
    bit27:
            brclr SCLPIN,PTAD,bit27;    // wait for SCL=1, then read SDA
            brclr SDAPIN,PTAD,bit27a;
            add #$01;                   // read SDA = 1
    bit27a:     
            lsla;                       // read SDA = 0
    w27:
            brset SCLPIN,PTAD,w27;      // wait for SCL = 0;
            //--------- receive bit6 ----------------------
    bit26:
            brclr SCLPIN,PTAD,bit26;
            brclr SDAPIN,PTAD,bit26a;
            add #$01;                   // read SDA = 1
    bit26a:
            lsla;                       // read SDA = 0
    w26:
            brset SCLPIN,PTAD,w26;
            //--------- receive bit5 ----------------------
    bit25:
            brclr SCLPIN,PTAD,bit25;
            brclr SDAPIN,PTAD,bit25a;
            add #$01;                   // read SDA = 1
    bit25a:
            lsla;                       // read SDA = 0
    w25:    
            brset SCLPIN,PTAD,w25;
            //--------- receive bit4 ----------------------
    bit24:
            brclr SCLPIN,PTAD,bit24;
            brclr SDAPIN,PTAD,bit24a;
            add #$01;                   // read SDA = 1
    bit24a:
            lsla;                       // read SDA = 0
    w24:    
            brset SCLPIN,PTAD,w24;
            //--------- receive bit3 ----------------------
    bit23:
            brclr SCLPIN,PTAD,bit23;
            brclr SDAPIN,PTAD,bit23a;
            add #$01;                   // read SDA = 1
    bit23a:
            lsla;                       // read SDA = 0
    w23:
            brset SCLPIN,PTAD,w23;
            //--------- receive bit2 ----------------------
    bit22:
            brclr SCLPIN,PTAD,bit22;
            brclr SDAPIN,PTAD,bit22a;
            add #$01;                   // read SDA = 1
    bit22a:
            lsla;                       // read SDA = 0
    w22:
            brset SCLPIN,PTAD,w22;
            //--------- receive bit1 ----------------------
    bit21:
            brclr SCLPIN,PTAD,bit21;
            brclr SDAPIN,PTAD,bit21a;
            add #$01;                   // read SDA = 1
    bit21a:
            lsla;                       // read SDA = 0
    w21:
            brset SCLPIN,PTAD,w21;
            //--------- receive bit0 ----------------------
    bit20:
            brclr SCLPIN,PTAD,bit20;
            brclr SDAPIN,PTAD,w20;
            add #$01;                   // read SDA = 1
    w20:
            brset SCLPIN,PTAD,w20;
            //--------- end of receive second byte ------------------
            
            //--------- send SL_ACK -----------------------            
            bclr SDAPIN,PTAD;
            bset SDAPIN,PTADD;          // put 0 to SDA -> SL_ACK;
    sl_ack_2a:
            brclr SCLPIN,PTAD,sl_ack_2a;
            sta Command;                // save received command byte;
    sl_ack_2b:
            brset SCLPIN,PTAD,sl_ack_2b;
            bclr SDAPIN,PTADD;          // SDA as input;
            //--------- receive STOP --------------------------------
    fin_stop:
            lda PTAD;
            and #I2C_MASK;
            cmp #I2C_MASK;
            bne fin_stop;           // STOP condition occurred / SDA=SCL=1 /;
            //---------- end of IIC receive procedure ---------------
            RTS;

//-----------------------------------------------------------------------------
            //---------- Send data to Master ------------------------
    send_data:   
            bset SDAPIN,PTADD;          // SDA as output;
            //--------- send bit7 -------------------------
            brclr 7,TX_data,tx07;
            bset SDAPIN,PTAD;
    tx07:
            brclr SCLPIN,PTAD,tx07;
    tx07a:
            brset SCLPIN,PTAD,tx07a;
            bclr SDAPIN,PTAD;
            //--------- send bit6 -------------------------
            brclr 6,TX_data,tx06
            bset SDAPIN,PTAD;
    tx06:
            brclr SCLPIN,PTAD,tx06;
    tx06a:
            brset SCLPIN,PTAD,tx06a;
            bclr SDAPIN,PTAD;
            //--------- send bit5 -------------------------
            brclr 5,TX_data,tx05
            bset SDAPIN,PTAD;
    tx05:
            brclr SCLPIN,PTAD,tx05;
    tx05a:
            brset SCLPIN,PTAD,tx05a;
            bclr SDAPIN,PTAD;
            //--------- send bit4 -------------------------
            brclr 4,TX_data,tx04
            bset SDAPIN,PTAD;
    tx04:
            brclr SCLPIN,PTAD,tx04;
    tx04a:
            brset SCLPIN,PTAD,tx04a;
            bclr SDAPIN,PTAD;
            //--------- send bit3 -------------------------
            brclr 3,TX_data,tx03
            bset SDAPIN,PTAD;
    tx03:
            brclr SCLPIN,PTAD,tx03;
    tx03a:
            brset SCLPIN,PTAD,tx03a;
            bclr SDAPIN,PTAD;
            //--------- send bit2 -------------------------
            brclr 2,TX_data,tx02
            bset SDAPIN,PTAD;
    tx02:
            brclr SCLPIN,PTAD,tx02;
    tx02a:
            brset SCLPIN,PTAD,tx02a;
            bclr SDAPIN,PTAD;
            //--------- send bit1 -------------------------
            brclr 1,TX_data,tx01
            bset SDAPIN,PTAD;
    tx01:
            brclr SCLPIN,PTAD,tx01;
    tx01a:
            brset SCLPIN,PTAD,tx01a;
            bclr SDAPIN,PTAD;
            //--------- send bit0 -------------------------
            brclr 0,TX_data,tx00
            bset SDAPIN,PTAD;
    tx00:
            brclr SCLPIN,PTAD,tx00;
    tx00a:
            brset SCLPIN,PTAD,tx00a;
            bclr SDAPIN,PTAD;
            //--------- end of send data ----------------------------
            bclr SDAPIN,PTADD;          // SDA as input;

            //--------- MS_ACK Time gap -------------------            
    ms_ack_rda:
            brclr SCLPIN,PTAD,ms_ack_rda;
    ms_ack_rdb:
            brset SCLPIN,PTAD,ms_ack_rdb;
            //--------- end of SL_ACK -------------------------------
    w_stop1:
            lda PTAD;
            and #I2C_MASK;
            cmp #SCLPIN;                // test for SDA = 0, SCL = 1;
            bne w_stop1;
    fin_stop1:
            lda PTAD;
            and #I2C_MASK;
            cmp #I2C_MASK;
            bne fin_stop1;
            //--------- end of IIC communication ----------
    }
}
/********************************************************************************
*   Description: Receives RC-5 code;                                            *
*                                                                               *
*   Input:  IR RC-5 demodulated signal on RC5_PIN;                              *
*                                                                               *
*   Output: "Address" and "Command" bytes;                                      *
*                                                                               *
********************************************************************************/
void IR_receive(void)
{
    byte i;
    
    KBISC_KBIE = 0;                     // Disable KBI interrupt;
    KBISC_KBACK = 1;                    // Clear the flag;
    MTIMSC_TSTP = 0;                    // start counting;
    IR_data = 0;
    //;--------- receive first bit(S0) --------------------
    Delay(245);
    if(!RC5_PIN) IR_data |= 0x01;       // read RC5 = 1;
    IR_data <<= 1;
    //;--------- receive (S1,T0) + Address (A4 - A0) ------
    for(i=7;i>0;i--)
    {
        Delay(250);
        Delay(250);
        Delay(250);
        Delay(200);
        if(!RC5_PIN) IR_data |= 0x01;   // read RC5 = 1;
        IR_data <<= 1;
    }
    Address = (IR_data >> 1) & 0x1F;
    IR_data = 0;
    //;--------- receive Command bits(D5 - D0) ------------
    for(i=6;i>0;i--)
    {
        Delay(250);
        Delay(250);
        Delay(250);
        Delay(200);
        if(!RC5_PIN) IR_data |= 0x01;   // read RC5 = 1;
        IR_data <<= 1;
    }
    Command = (IR_data >> 1) & 0x3F;
    IR_data = 0;
}
/********************************************************************************
*   Description: Senses the user button;                                        *
*                                                                               *
*   Input:  STBY button pressed;                                                *
*                                                                               *
*   Output: variable:  mode;                                                    *
*                                                                               *
********************************************************************************/
void STBY_Button(void)  /* service the STBY button in RUN or SLEEP mode */
{
    WaitNms(150);
    if (!BUTTON) while (!BUTTON) __RESET_WATCHDOG();;
    if (mode == RUN)
    {
        PWR_PIN = OFF;
        mode = SLEEP;
    }
    else if (mode == SLEEP)
    {
        PWR_PIN = ON;
        mode = RUN;
    }
}
/********************************************************************************
*   Description: Puts MCU into low power STOP mode;                             *
*                                                                               *
*   Input:  variable:    mode == SLEEP;                                         *
*                                                                               *
*   Output: MCU in STOP mode;                                                   *
*                                                                               *
********************************************************************************/
void Go_sleep(void)                     // Set MCU before sleep;
{
        /* Set the MTIM module for IR receive after wake-up */
        MTIMMOD = 0xFE;                 // for 8MHz BusCLK;
        bit_length = 0x0D;              // 1/4 of the bit length of the RC-5 code;
        MTIMCLK = 0;                    // BusCLK / 1 as clock source for MTIM;
        MTIMSC_TSTP = 1;                // STOP timer;
        MTIMSC_TRST = 1;                // RESET timer;
        
        /* Set active pins for wake-up from sleep mode */
        KBISC_KBIE = 0;
        KBIPE = (RC5_PIN_MASK | BUTTON_MASK);   // Enable wake-up 
                                                // by IR or STBY button;
        KBISC_KBACK = 1;                        // Clear the KBI flag;
        KBISC_KBIE = 1;                         // Enable KBI interrupt;
}
/********************************************************************************
*   Time delay functions                                                        *
********************************************************************************/
void Delay(byte del)                    // Universal delay;
{
    byte i;
    i = del;
    while(i > 0)
    {
        i--;
        __RESET_WATCHDOG();
    }
}
//-----------------------------------------------------------------------------
void Wait1ms(void)                      // for 8MHz BusCLK;
{
    Delay(250);
    Delay(250);
    Delay(25);
}
//-----------------------------------------------------------------------------
void WaitNms(byte del)                  // for 8MHz BusCLK;
{
    byte i;
    i = del;
    while(i > 0) 
    {
        Wait1ms();
        i--;
    }
}
/********************************************************************************
*   Description: set initial values of variebles;                               *
*                                                                               *
*   Input:  NO;                                                                 *
*                                                                               *
*   Output: default values for variables;                                       *
*                                                                               *
********************************************************************************/
void Init_Variables(void)
{
    PWR_PIN = ON;
    count = 0;
    temp = 0;
    mode = RUN;
    command_ON = 0;
    command_OFF = 0;
    r_w = 0;
    RX_data = 0;
    TX_data = 0;
    /* Set the MTIM module for IR receive */
    MTIMMOD = 0xFE;                     // for 8MHz BusCLK;
    bit_length = 0x0D;                  // 1/4 of the bit length of the RC-5 code;
    MTIMCLK = 0;                        // BusCLK / 1 as clock source for MTIM;
    MTIMSC_TSTP = 1;                    // STOP timer;
    MTIMSC_TRST = 1;                    // RESET timer;
}
/********************************************************************************
*   Description: Compares received IR data;                                     *
*                                                                               *
*   Input:  variables "Address" and "Command";                                  *
*                                                                               *
*   Output: variables: - address_valid, command_ON, command_OFF;                *
*                                                                               *
********************************************************************************/
void Compare_IR(void)
{
    address_valid = FALSE;
    switch (Address)
    {
        case ADDR1:
        case ADDR2:
        case ADDR3:
        case ADDR4:
        case ADDR5:
        case ADDR6:
        case ADDR7:
        case ADDR8:
        case ADDR9:
        case ADDR10:
        case ADDR11:
        case ADDR12:
        {
            address_valid = TRUE;
        } break;
    }
    
    command_ON = FALSE;
    switch (Command)
    {
        case CMD_ON1:
        case CMD_ON2:
        case CMD_ON3:
        case CMD_ON4:
        case CMD_ON5:
        case CMD_ON6:
        case CMD_ON7:
        case CMD_ON8:
        case CMD_ON9:
        case CMD_ON10:
        case CMD_ON11:
        {
            command_ON = TRUE;
        } break;
    }
    
    command_OFF = FALSE;
    switch (Command)
    {
        case CMD_OFF1:
        {
            command_OFF = TRUE;
        } break;
    }
}
/********************************************************************************
********************************************************************************/
